<?php
/**
 * Plugin Name: Lean Bunker – Security Suite (Safe Mode)
 * Plugin URI: https://universalopus.com/
 * Description: Versione sicura per tutti: firewall leggero, rimozione bloat, disabilitazione Gutenberg, protezione utenti. Niente CSP, COEP, o header rischiosi.
 * Version: 1.0.8-safe
 * Author: Riccardo Bastillo
 * Author URI: https://universalopus.com/
 * License: GPL-3.0+
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 * Network: true
 */

defined('ABSPATH') || exit;

// ───────────────────────────────────────
// 🧠 UTILITIES CONDIVISE (minimale)
// ───────────────────────────────────────

if (!function_exists('lean_bunker_get_ip')) {
    function lean_bunker_get_ip() {
        $ip = '0.0.0.0';
        if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['HTTP_CF_CONNECTING_IP']));
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ips = explode(',', sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR'])));
            $ip = trim($ips[0]);
        } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR']));
        }
        return $ip;
    }
}

// ───────────────────────────────────────
// ⚙️ CONFIGURAZIONE
// ───────────────────────────────────────

if (!function_exists('lean_bunker_get_option')) {
    function lean_bunker_get_option($key, $default = false) {
        $options = get_site_option('lean_bunker_safe_settings', array());
        return isset($options[$key]) ? (bool) $options[$key] : $default;
    }
}

// ───────────────────────────────────────
// 🔧 ADMIN UI (solo opzioni sicure)
// ───────────────────────────────────────

if (is_admin()) {
    add_action('network_admin_menu', 'lean_bunker_safe_admin_menu');
    add_action('admin_menu', 'lean_bunker_safe_admin_menu_single');
    add_action('admin_init', 'lean_bunker_safe_save_settings');

    function lean_bunker_safe_admin_menu() {
        add_submenu_page('settings.php', 'Lean Bunker (Safe)', 'Lean Bunker (Safe)', 'manage_options', 'lean-bunker-safe', 'lean_bunker_safe_admin_page');
    }

    function lean_bunker_safe_admin_menu_single() {
        if (!is_multisite()) {
            add_options_page('Lean Bunker (Safe)', 'Lean Bunker (Safe)', 'manage_options', 'lean-bunker-safe', 'lean_bunker_safe_admin_page');
        }
    }

    function lean_bunker_safe_save_settings() {
        if (!isset($_POST['lean_bunker_nonce']) || !wp_verify_nonce($_POST['lean_bunker_nonce'], 'lean_bunker_save') || !current_user_can('manage_options')) {
            return;
        }
        $options = array(
            'disable_gutenberg' => !empty($_POST['disable_gutenberg']),
            'disable_heartbeat' => !empty($_POST['disable_heartbeat']),
            'disable_xmlrpc' => !empty($_POST['disable_xmlrpc']),
            'disable_feeds' => !empty($_POST['disable_feeds']),
            'disable_comments' => !empty($_POST['disable_comments']),
            'remove_bloat' => !empty($_POST['remove_bloat']),
            'disable_jquery' => !empty($_POST['disable_jquery']),
            'isolate_user' => !empty($_POST['isolate_user']),
            'firewall' => !empty($_POST['firewall']),
        );
        update_site_option('lean_bunker_safe_settings', $options);
        add_action('admin_notices', function() {
            echo '<div class="notice notice-success"><p>Impostazioni salvate.</p></div>';
        });
    }

    function lean_bunker_safe_admin_page() {
        ?>
        <div class="wrap">
            <h1>Lean Bunker – Security Suite (Safe Mode)</h1>
            <p>✅ Questa versione <strong>non blocca mai il sito</strong>, anche con Beaver Builder, Elementor o altri page builder.</p>
            <form method="post">
                <?php wp_nonce_field('lean_bunker_save', 'lean_bunker_nonce'); ?>
                <h2>🧹 Rimozione bloat (sicuro)</h2>
                <table class="form-table">
                    <tr><th scope="row">Disabilita Gutenberg</th><td><label><input type="checkbox" name="disable_gutenberg" <?php checked(lean_bunker_get_option('disable_gutenberg')); ?>> Abilita</label></td></tr>
                    <tr><th scope="row">Disabilita Heartbeat</th><td><label><input type="checkbox" name="disable_heartbeat" <?php checked(lean_bunker_get_option('disable_heartbeat')); ?>> Abilita</label></td></tr>
                    <tr><th scope="row">Disabilita XML-RPC</th><td><label><input type="checkbox" name="disable_xmlrpc" <?php checked(lean_bunker_get_option('disable_xmlrpc')); ?>> Abilita</label></td></tr>
                    <tr><th scope="row">Disabilita Feed RSS</th><td><label><input type="checkbox" name="disable_feeds" <?php checked(lean_bunker_get_option('disable_feeds')); ?>> Abilita</label></td></tr>
                    <tr><th scope="row">Disabilita Commenti</th><td><label><input type="checkbox" name="disable_comments" <?php checked(lean_bunker_get_option('disable_comments')); ?>> Abilita</label></td></tr>
                    <tr><th scope="row">Rimuovi bloat (emoji, embed, ecc.)</th><td><label><input type="checkbox" name="remove_bloat" <?php checked(lean_bunker_get_option('remove_bloat')); ?>> Abilita</label></td></tr>
                    <tr><th scope="row">Rimuovi jQuery in frontend</th><td><label><input type="checkbox" name="disable_jquery" <?php checked(lean_bunker_get_option('disable_jquery')); ?>> Abilita (solo se non usi temi/plugin che ne dipendono)</label></td></tr>
                    <tr><th scope="row">Isolamento utente (limita accesso ad altri profili)</th><td><label><input type="checkbox" name="isolate_user" <?php checked(lean_bunker_get_option('isolate_user')); ?>> Abilita</label></td></tr>
                    <tr><th scope="row">Firewall leggero (blocco brute-force e bot)</th><td><label><input type="checkbox" name="firewall" <?php checked(lean_bunker_get_option('firewall')); ?>> Abilita</label></td></tr>
                </table>
                <p class="submit"><input type="submit" class="button-primary" value="Salva Impostazioni"></p>
            </form>

            <h2>🔍 Stato</h2>
            <ul>
                <li><strong>Gutenberg disattivato:</strong> <?php echo lean_bunker_get_option('disable_gutenberg') ? 'Sì' : 'No'; ?></li>
                <li><strong>Commenti disattivati:</strong> <?php echo lean_bunker_get_option('disable_comments') ? 'Sì' : 'No'; ?></li>
                <li><strong>Multisite:</strong> <?php echo is_multisite() ? 'Sì' : 'No'; ?></li>
            </ul>
        </div>
        <?php
    }
}

// ───────────────────────────────────────
// 🔥 FIREWALL LEGGERO (solo login + bot blacklist)
// ───────────────────────────────────────

if (lean_bunker_get_option('firewall')) {
    // Blocco brute-force
    add_action('wp_login_failed', 'lean_bunker_safe_login_failed');
    function lean_bunker_safe_login_failed($username) {
        $ip = lean_bunker_get_ip();
        $key = 'lean_bunker_login_' . md5($ip);
        $count = (int) get_transient($key);
        set_transient($key, $count + 1, 5 * MINUTE_IN_SECONDS);
        if ($count >= 3) {
            // Blocco temporaneo: disabilita login per 15 min
            set_transient('lean_bunker_blocked_ip_' . md5($ip), 1, 15 * MINUTE_IN_SECONDS);
        }
    }

    add_action('login_init', 'lean_bunker_safe_check_blocked_ip');
    function lean_bunker_safe_check_blocked_ip() {
        $ip = lean_bunker_get_ip();
        if (get_transient('lean_bunker_blocked_ip_' . md5($ip))) {
            wp_die('Accesso temporaneamente bloccato per troppi tentativi.', 'Accesso bloccato', array('response' => 429));
        }
    }

    // Blocco bot malevoli noti
    add_action('init', 'lean_bunker_safe_block_bad_bots', -5);
    function lean_bunker_safe_block_bad_bots() {
        if (is_admin() || is_user_logged_in()) return;
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $bad_bots = array('sqlmap', 'nikto', 'hydra', 'nmap', 'acunetix', 'burpsuite', 'nessus', 'metasploit', 'gobuster', 'c99', 'r57', 'webshell');
        foreach ($bad_bots as $bot) {
            if (stripos($ua, $bot) !== false) {
                wp_die('Accesso non autorizzato.', 'Bot bloccato', array('response' => 403));
            }
        }
    }
}

// ───────────────────────────────────────
// 👤 ISOLAMENTO UTENTE (sicuro)
// ───────────────────────────────────────

if (lean_bunker_get_option('isolate_user')) {
    add_action('pre_get_posts', 'lean_bunker_safe_isolate_author');
    function lean_bunker_safe_isolate_author($q) {
        if (is_admin() || !$q->is_main_query() || !is_user_logged_in() || current_user_can('manage_options')) return;
        if ($q->is_author && (int)$q->get('author') !== get_current_user_id()) {
            wp_safe_redirect(home_url());
            exit;
        }
    }

    add_action('load-profile.php', 'lean_bunker_safe_restrict_profile');
    add_action('load-user-edit.php', 'lean_bunker_safe_restrict_profile');
    function lean_bunker_safe_restrict_profile() {
        if (!is_user_logged_in() || current_user_can('manage_options')) return;
        if (isset($_GET['user_id']) && (int)$_GET['user_id'] !== get_current_user_id()) {
            wp_safe_redirect(admin_url('profile.php'));
            exit;
        }
    }
}

// ───────────────────────────────────────
// 🧹 BLOAT REMOVAL (sicuro, niente side-effect)
// ───────────────────────────────────────

if (lean_bunker_get_option('disable_gutenberg')) {
    add_filter('use_block_editor_for_post_type', '__return_false', 100);
}

if (lean_bunker_get_option('disable_heartbeat')) {
    add_action('init', function() {
        if (!is_admin()) wp_deregister_script('heartbeat');
    });
    add_filter('heartbeat_settings', function($settings) {
        $settings['interval'] = 120;
        return $settings;
    });
}

if (lean_bunker_get_option('disable_xmlrpc')) {
    add_filter('xmlrpc_enabled', '__return_false');
    add_filter('pre_option_enable_xmlrpc', '__return_zero');
}

if (lean_bunker_get_option('disable_feeds')) {
    foreach (array('do_feed', 'do_feed_rdf', 'do_feed_rss', 'do_feed_rss2', 'do_feed_atom') as $f) {
        add_action($f, function() { status_header(410); exit; }, 1);
    }
}

if (lean_bunker_get_option('disable_comments')) {
    add_action('admin_init', function() {
        foreach (get_post_types() as $type) {
            remove_post_type_support($type, 'comments');
            remove_post_type_support($type, 'trackbacks');
        }
    });
    add_filter('comments_open', '__return_false', 20);
    add_filter('pings_open', '__return_false', 20);
    add_action('admin_menu', function() { remove_menu_page('edit-comments.php'); });
}

if (lean_bunker_get_option('remove_bloat')) {
    remove_action('wp_head', 'print_emoji_detection_script', 7);
    remove_action('wp_print_styles', 'print_emoji_styles');
    remove_action('wp_head', 'wp_oembed_add_discovery_links');
    remove_action('wp_head', 'wp_generator');
    remove_action('wp_head', 'rsd_link');
    remove_action('wp_head', 'wlwmanifest_link');
    remove_action('wp_head', 'rest_output_link_wp_head');
    remove_action('wp_print_styles', 'print_emoji_styles');
    remove_filter('the_content_feed', 'wp_staticize_emoji');
    remove_action('admin_print_scripts', 'print_emoji_detection_script');
}

if (lean_bunker_get_option('disable_jquery')) {
    add_action('wp_enqueue_scripts', function() {
        if (!is_user_logged_in()) {
            wp_dequeue_script('jquery');
            wp_dequeue_script('jquery-core');
            wp_dequeue_script('jquery-migrate');
        }
    }, 100);
}