<?php
/**
 * Plugin Name: Lean Bunker SEO Checker
 * Description: Verifica granulare e oggettiva dell'ottimizzazione SEO lato server e WordPress. Punteggio 0-100 e report dettagliato.
 * Version: 1.0.1
 * Author: Riccardo
 * License: GPL-3.0+
 */

if (!defined('ABSPATH')) {
    exit;
}

// 🔒 Carica solo in admin (niente frontend, niente shortcode)
if (defined('WP_CLI') && WP_CLI) {
    return;
} elseif (defined('DOING_AJAX') && DOING_AJAX) {
    return;
} elseif (!is_admin()) {
    return; // non caricare in frontend
}

// Da qui in poi: solo admin
add_action('admin_menu', 'seo_checker_add_admin_page');
function seo_checker_add_admin_page() {
    if (!current_user_can('manage_options')) {
        return;
    }
    add_management_page(
        'SEO Checker',
        'SEO Checker',
        'manage_options',
        'wp-seo-check',
        'seo_checker_render_admin_page'
    );
}

add_action('admin_enqueue_scripts', 'seo_checker_enqueue_scripts');
function seo_checker_enqueue_scripts($hook) {
    if ($hook !== 'tools_page_wp-seo-check') {
        return;
    }
    wp_enqueue_style('wp-seo-check-style', false);
    $css = '
        .seo-check { margin: 6px 0; padding-left: 25px; font-family: monospace; }
        .seo-pass { color: #008000; }
        .seo-fail { color: #d63638; }
        .seo-info { color: #666; font-size: 0.9em; }
        .seo-score { font-size: 26px; font-weight: bold; margin: 20px 0; color: #2271b1; }
        .seo-section { margin-top: 25px; padding-top: 20px; border-top: 2px solid #f0f0f1; }
        .seo-subsection { margin-top: 15px; margin-bottom: 8px; font-weight: 600; color: #1d2327; }
    ';
    wp_add_inline_style('wp-seo-check-style', $css);
}

function seo_checker_render_admin_page() {
    if (!isset($_GET['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'seo_run_check')) {
        $nonce = wp_create_nonce('seo_run_check');
        echo '<div class="wrap">';
        echo '<h1>🔍 Lean Bunker SEO Checker</h1>';
        echo '<p>Analisi oggettiva della visibilità e indicizzabilità del sito da parte dei motori di ricerca.</p>';
        echo '<a href="' . esc_url(add_query_arg('nonce', $nonce)) . '" class="button button-primary">Esegui l\'analisi SEO</a>';
        echo '</div>';
        return;
    }

    $report = seo_checker_run_full_check();
    $score  = $report['score'];

    // Rimossi i salvataggi per badge pubblico
    update_option('seo_checker_last_check_time', time());

    echo '<div class="wrap">';
    echo '<h1>🔍 Lean Bunker SEO Checker</h1>';
    echo '<div class="seo-score">Punteggio SEO: <strong>' . esc_html($score) . '/100</strong></div>';

    if ($score >= 90) {
        echo '<div class="notice notice-success"><p>✅ Eccellente visibilità per i motori di ricerca.</p></div>';
    } elseif ($score >= 70) {
        echo '<div class="notice notice-warning"><p>⚠️ Buona visibilità, ma alcuni miglioramenti sono possibili.</p></div>';
    } else {
        echo '<div class="notice notice-error"><p>❌ Visibilità compromessa. I motori di ricerca potrebbero non indicizzare correttamente il sito.</p></div>';
    }

    foreach ($report['sections'] as $section_name => $subsections) {
        echo '<div class="seo-section"><h2>' . esc_html($section_name) . '</h2>';
        foreach ($subsections as $subsection_name => $checks) {
            echo '<div class="seo-subsection">' . esc_html($subsection_name) . '</div>';
            foreach ($checks as $check) {
                $class = $check['status'] === 'pass' ? 'seo-pass' : ($check['status'] === 'fail' ? 'seo-fail' : 'seo-info');
                echo '<div class="seo-check ' . esc_attr($class) . '">● ' . esc_html($check['name']) . ': ' . esc_html($check['message']) . '</div>';
            }
        }
        echo '</div>';
    }

    $nonce = wp_create_nonce('seo_run_check');
    echo '<p><a href="' . esc_url(add_query_arg('nonce', $nonce)) . '" class="button">Riesegui l\'analisi</a></p>';
    echo '</div>';
}

// ───────────────────────────────────────
// 🔁 TUTTE LE FUNZIONI DI CHECK
// ───────────────────────────────────────

function seo_checker_run_full_check() {
    $sections = array();

    $sections['🌐 Accessibilità & Server']      = seo_checker_check_accessibility();
    $sections['HTTPHeader']                     = seo_checker_check_headers();
    $sections['📄 robots.txt & Sitemap']        = seo_checker_check_robots_and_sitemap();
    $sections['💻 Frontend & Contenuti']        = seo_checker_check_frontend();
    $sections['🖼️ Asset & Prestazioni']         = seo_checker_check_assets();
    $sections['🔗 Struttura URL']               = seo_checker_check_url_structure();

    $total_score = 0;
    foreach ($sections as $subsection_groups) {
        foreach ($subsection_groups as $checks) {
            foreach ($checks as $check) {
                if (isset($check['points']) && $check['status'] === 'pass') {
                    $total_score += $check['points'];
                }
            }
        }
    }

    $final_score = min(100, max(0, (int) round($total_score)));
    return array('score' => $final_score, 'sections' => $sections);
}

function seo_checker_check_accessibility() {
    $checks = array();
    $home_url = home_url('/');
    $home_resp = wp_remote_get($home_url, array('timeout' => 8, 'sslverify' => false));
    $home_code = wp_remote_retrieve_response_code($home_resp);
    $home_ok = ($home_code === 200);
    $checks['Accesso base']['Homepage'] = array(
        'name' => 'Homepage accessibile',
        'status' => $home_ok ? 'pass' : 'fail',
        'message' => $home_ok ? '200 OK' : 'Errore HTTP ' . $home_code,
        'points' => $home_ok ? 10 : 0
    );

    $is_https = (is_ssl() || strpos(home_url(), 'https://') === 0);
    $checks['Accesso base']['HTTPS'] = array(
        'name' => 'Connessione HTTPS',
        'status' => $is_https ? 'pass' : 'fail',
        'message' => $is_https ? 'Attivo' : 'Mancante (obbligatorio per SEO moderna)',
        'points' => $is_https ? 8 : 0
    );

    return $checks;
}

function seo_checker_check_headers() {
    $url = home_url('/');
    $response = wp_remote_get($url, array('timeout' => 8));
    $headers = wp_remote_retrieve_headers($response);
    $x_robots = isset($headers['x-robots-tag']) ? $headers['x-robots-tag'] : '';
    $noindex_in_xrobots = (stripos($x_robots, 'noindex') !== false);
    $content_type = isset($headers['content-type']) ? $headers['content-type'] : '';
    $html_content = (stripos($content_type, 'text/html') !== false);

    return array(
        'Headers critici' => array(
            'X-Robots-Tag' => array(
                'name' => 'Header X-Robots-Tag',
                'status' => !$noindex_in_xrobots ? 'pass' : 'fail',
                'message' => !$noindex_in_xrobots ? 'Assente o permissivo' : 'Contiene "noindex"',
                'points' => !$noindex_in_xrobots ? 6 : 0
            ),
            'Content-Type' => array(
                'name' => 'Content-Type HTML',
                'status' => $html_content ? 'pass' : 'fail',
                'message' => $html_content ? 'Corretto' : 'Non HTML (' . esc_html($content_type) . ')',
                'points' => $html_content ? 4 : 0
            )
        )
    );
}

function seo_checker_check_robots_and_sitemap() {
    $checks = array();

    $robots_url = home_url('/robots.txt');
    $robots_resp = wp_remote_get($robots_url, array('timeout' => 5));
    $robots_code = wp_remote_retrieve_response_code($robots_resp);
    $robots_body = wp_remote_retrieve_body($robots_resp);
    $robots_ok = ($robots_code === 200 && strlen($robots_body) > 10);
    $robots_disallow_all = (strpos($robots_body, 'Disallow: /') !== false);

    $checks['File critici']['robots.txt'] = array(
        'name' => 'robots.txt accessibile',
        'status' => $robots_ok ? 'pass' : 'fail',
        'message' => $robots_ok ? '200 OK' : 'Non trovato o vuoto (HTTP ' . $robots_code . ')',
        'points' => $robots_ok ? 6 : 0
    );

    if ($robots_ok && $robots_disallow_all) {
        $checks['File critici']['robots.txt blocco totale'] = array(
            'name' => 'robots.txt blocco totale',
            'status' => 'fail',
            'message' => 'Contiene "Disallow: /" → blocca tutti i crawler',
            'points' => 0
        );
    }

    $sitemap_urls = array(home_url('/sitemap.xml'), home_url('/wp-sitemap.xml'));
    $sitemap_found = $sitemap_valid = false;
    $sitemap_url_used = '';

    foreach ($sitemap_urls as $url) {
        $resp = wp_remote_get($url, array('timeout' => 6));
        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        if ($code === 200 && (strpos($body, '<urlset') !== false || strpos($body, '<sitemapindex') !== false)) {
            $sitemap_found = $sitemap_valid = true;
            $sitemap_url_used = $url;
            break;
        } elseif ($code === 200 && !empty($body)) {
            $sitemap_found = true;
            $sitemap_url_used = $url;
        }
    }

    $checks['File critici']['Sitemap'] = array(
        'name' => 'Sitemap XML accessibile',
        'status' => $sitemap_found ? 'pass' : 'fail',
        'message' => $sitemap_found ? 'Trovata (' . esc_url($sitemap_url_used) . ')' : 'Non trovata',
        'points' => $sitemap_valid ? 8 : ($sitemap_found ? 4 : 0)
    );

    return $checks;
}

function seo_checker_check_frontend() {
    $url = home_url('/');
    $response = wp_remote_get($url, array('timeout' => 8));
    $body = wp_remote_retrieve_body($response);

    $title = '';
    if (preg_match('#<title>(.*?)</title>#si', $body, $matches)) {
        $title = trim(wp_strip_all_tags($matches[1]));
    }
    $title_ok = (!empty($title) && strlen($title) <= 70);

    $desc = '';
    if (preg_match('#<meta\s+name=["\']description["\']\s+content=["\'](.*?)["\']#si', $body, $matches)) {
        $desc = trim(wp_strip_all_tags($matches[1]));
    }
    $desc_ok = (!empty($desc) && strlen($desc) >= 50 && strlen($desc) <= 160);

    $canonical_ok = (preg_match('#<link\s+rel=["\']canonical["\']\s+href=["\']' . preg_quote(home_url(), '/') . '#si', $body));
    $viewport_ok = (stripos($body, 'name="viewport"') !== false);
    $has_noindex_org = (stripos($body, 'name="robots"') !== false && stripos($body, 'noindex') !== false);

    return array(
        'Meta e struttura' => array(
            'Title tag' => array('name' => 'Title tag', 'status' => $title_ok ? 'pass' : 'fail', 'message' => $title_ok ? 'Presente e valido' : 'Mancante o troppo lungo', 'points' => $title_ok ? 8 : 0),
            'Meta description' => array('name' => 'Meta description', 'status' => $desc_ok ? 'pass' : 'info', 'message' => $desc_ok ? 'Presente e ottimale' : (!empty($desc) ? 'Presente ma non ottimale' : 'Mancante'), 'points' => $desc_ok ? 6 : 0),
            'Canonical' => array('name' => 'Canonical URL', 'status' => $canonical_ok ? 'pass' : 'info', 'message' => $canonical_ok ? 'Presente e corretto' : 'Mancante o non corrispondente', 'points' => $canonical_ok ? 4 : 0),
            'Meta robots noindex' => array('name' => 'Meta robots noindex', 'status' => !$has_noindex_org ? 'pass' : 'fail', 'message' => !$has_noindex_org ? 'Assente' : 'Presente → blocca indicizzazione', 'points' => !$has_noindex_org ? 6 : 0)
        ),
        'Mobile' => array(
            'Viewport tag' => array('name' => 'Viewport per mobile', 'status' => $viewport_ok ? 'pass' : 'fail', 'message' => $viewport_ok ? 'Presente' : 'Mancante → non mobile-friendly', 'points' => $viewport_ok ? 6 : 0)
        )
    );
}

function seo_checker_check_assets() {
    $url = home_url('/');
    $response = wp_remote_get($url, array('timeout' => 8));
    $body = wp_remote_retrieve_body($response);

    $css_ok = (preg_match('#<link[^>]+href=["\'][^"\']*\.css#i', $body));
    $js_ok = (preg_match('#<script[^>]+src=["\'][^"\']*\.js#i', $body));

    $uploads_test = home_url('/wp-content/uploads/');
    $uploads_resp = wp_remote_get($uploads_test, array('timeout' => 4));
    $uploads_code = wp_remote_retrieve_response_code($uploads_resp);
    $uploads_blocked = ($uploads_code === 403 || $uploads_code === 404);

    return array(
        'Risorse' => array(
            'CSS caricato' => array('name' => 'CSS caricato', 'status' => $css_ok ? 'pass' : 'fail', 'message' => $css_ok ? 'Presente' : 'Mancante → Google non può renderizzare', 'points' => $css_ok ? 6 : 0),
            'JavaScript caricato' => array('name' => 'JavaScript caricato', 'status' => $js_ok ? 'pass' : 'info', 'message' => $js_ok ? 'Presente' : 'Mancante (non critico se contenuto statico)', 'points' => $js_ok ? 2 : 0),
            'Cartella uploads' => array('name' => 'Cartella /uploads/ accessibile', 'status' => !$uploads_blocked ? 'pass' : 'info', 'message' => !$uploads_blocked ? 'Accessibile' : 'Bloccata (potrebbe impedire indicizzazione immagini)', 'points' => !$uploads_blocked ? 4 : 0)
        )
    );
}

function seo_checker_check_url_structure() {
    $permalinks = get_option('permalink_structure');
    $pretty_permalinks = !empty($permalinks) && $permalinks !== '/?p=%post_id%';
    $home_has_params = (parse_url(home_url(), PHP_URL_QUERY) !== null);

    return array(
        'URL' => array(
            'Permalink puliti' => array('name' => 'Permalink SEO-friendly', 'status' => $pretty_permalinks ? 'pass' : 'fail', 'message' => $pretty_permalinks ? 'Attivi' : 'Disattivati (usa ?p=123)', 'points' => $pretty_permalinks ? 8 : 0),
            'Homepage pulita' => array('name' => 'Homepage senza parametri', 'status' => !$home_has_params ? 'pass' : 'info', 'message' => !$home_has_params ? 'URL pulito' : 'Contiene parametri (non ideale)', 'points' => !$home_has_params ? 4 : 0)
        )
    );
}